<?php
/**
 * Plugin Name:       Mail2Post by Stocki
 * Plugin URI:        https://mail2post.stockmap.at
 * Description:       Erstellt automatisch WordPress-Beiträge aus E-Mails. Der Betreff wird zum Titel, der Text zum Inhalt und ein Bildanhang zum Beitragsbild.
 * Version:           1.9.0
 * Author:            Stocki
 * Author URI:        https://mail2post.stockmap.at
 * License:           GPL v2 or later
 * License URI:       https://stockmap.at/license
 * Text Domain:       email-to-post
 * Domain Path:       /languages
 */

// Sicherheitsabfrage: Direkten Zugriff auf die Datei verhindern
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// ==============
// ABHÄNGIGKEITS-CHECK
// ==============

// Überprüft, ob die IMAP-Erweiterung vorhanden ist und zeigt ggf. eine Admin-Warnung an.
function etp_check_dependencies() {
    if ( ! function_exists( 'imap_open' ) ) {
        add_action( 'admin_notices', 'etp_show_imap_error_notice' );
    }
}
add_action( 'admin_init', 'etp_check_dependencies' );

function etp_show_imap_error_notice() {
    $settings_url = admin_url('edit.php?page=email-to-post-settings');
    ?>
    <div class="notice notice-error">
        <p>
            <strong><?php _e( 'E-Mail zu Beitrag - Aktion erforderlich:', 'email-to-post' ); ?></strong>
            <?php printf(
                __( 'Die PHP IMAP-Erweiterung fehlt. Das Plugin kann so nicht funktionieren. Bitte besuchen Sie die <a href="%s">System-Check Seite</a> für eine einfache Anleitung zur Lösung des Problems.', 'email-to-post' ),
                esc_url($settings_url . '#tab-check')
            ); ?>
        </p>
    </div>
    <?php
}


// ==============
// PLUGIN-AKTIVIERUNG & DEAKTIVIERUNG
// ==============

// Funktion, die bei der Aktivierung des Plugins ausgeführt wird
function etp_activate_plugin() {
    // Standardeinstellungen festlegen
    if ( ! get_option('etp_settings') ) {
        $default_settings = [
            'imap_host' => '',
            'imap_port' => '993',
            'imap_user' => '',
            'imap_pass' => '',
            'imap_encryption' => 'ssl',
            'allowed_emails' => [],
            'debug_mode' => false,
            'enable_modern_design' => true,
        ];
        update_option('etp_settings', $default_settings);
    }

    // Cron-Job planen, falls er noch nicht existiert
    if ( ! wp_next_scheduled( 'etp_check_email_hook' ) ) {
        wp_schedule_event( time(), 'every_five_minutes', 'etp_check_email_hook' );
    }
}
register_activation_hook( __FILE__, 'etp_activate_plugin' );

// Funktion, die bei der Deaktivierung des Plugins ausgeführt wird
function etp_deactivate_plugin() {
    // Geplanten Cron-Job entfernen
    wp_clear_scheduled_hook( 'etp_check_email_hook' );
}
register_deactivation_hook( __FILE__, 'etp_deactivate_plugin' );


// ==============
// CRON-JOB FÜR E-MAIL-ABRUF
// ==============

// Ein benutzerdefiniertes Intervall von 5 Minuten hinzufügen
function etp_add_cron_interval( $schedules ) {
    $schedules['every_five_minutes'] = array(
        'interval' => 300, // 5 * 60 Sekunden
        'display'  => esc_html__( 'Alle fünf Minuten' ),
    );
    return $schedules;
}
add_filter( 'cron_schedules', 'etp_add_cron_interval' );

// Die Hauptfunktion zum Abrufen und Verarbeiten von E-Mails an den Cron-Hook binden
add_action( 'etp_check_email_hook', 'etp_process_emails' );


// ==============
// HILFSFUNKTIONEN (DEBUG LOGGING)
// ==============
function etp_log($message) {
    $settings = get_option('etp_settings');
    if (isset($settings['debug_mode']) && $settings['debug_mode']) {
        $upload_dir = wp_upload_dir();
        $log_file = $upload_dir['basedir'] . '/email-to-post-debug.log';
        $timestamp = date('Y-m-d H:i:s');
        $log_message = "[$timestamp] " . (is_string($message) ? $message : print_r($message, true)) . "\n";
        file_put_contents($log_file, $log_message, FILE_APPEND);
    }
}

// ==============
// E-MAIL-VERARBEITUNGSLOGIK
// ==============

function etp_process_emails() {
    etp_log('Cron-Job gestartet.');
    if ( ! function_exists( 'imap_open' ) ) { 
        etp_log('FEHLER: IMAP-Funktion nicht vorhanden.');
        return; 
    }
    
    $settings = get_option('etp_settings');

    if ( empty($settings['imap_host']) || empty($settings['imap_user']) || empty($settings['imap_pass']) ) { 
        etp_log('WARNUNG: IMAP-Einstellungen unvollständig. Breche ab.');
        return; 
    }

    $host = $settings['imap_host'];
    $port = $settings['imap_port'];
    $user = $settings['imap_user'];
    $pass = $settings['imap_pass'];
    $encryption = $settings['imap_encryption'];
    $connection_string = '{' . $host . ':' . $port . '/imap/' . $encryption . '/novalidate-cert}INBOX';

    etp_log('Versuche, Verbindung herzustellen mit: ' . $connection_string);
    set_error_handler(function() { /* ignore errors */ });
    $inbox = imap_open( $connection_string, $user, $pass, 0, 1, ['DISABLE_AUTHENTICATOR' => 'GSSAPI'] );
    restore_error_handler();
    
    if ( ! $inbox ) { 
        etp_log('FEHLER: IMAP-Verbindung fehlgeschlagen. Letzter Fehler: ' . imap_last_error());
        return; 
    }

    etp_log('IMAP-Verbindung erfolgreich. Suche nach ungelesenen E-Mails.');
    $emails = imap_search( $inbox, 'UNSEEN' );

    if ( $emails ) {
        etp_log('Habe ' . count($emails) . ' ungelesene E-Mail(s) gefunden.');
        require_once( ABSPATH . 'wp-admin/includes/media.php' );
        require_once( ABSPATH . 'wp-admin/includes/file.php' );
        require_once( ABSPATH . 'wp-admin/includes/image.php' );

        foreach ( $emails as $email_number ) {
            etp_log("--- Verarbeite E-Mail Nr. {$email_number} ---");
            $header = imap_headerinfo( $inbox, $email_number );
            $from_address = strtolower( $header->from[0]->mailbox . '@' . $header->from[0]->host );
            etp_log('Absender: ' . $from_address);

            $allowed_emails = array_map('strtolower', $settings['allowed_emails']);
            if ( ! in_array( $from_address, $allowed_emails ) ) { 
                etp_log('WARNUNG: Absender ist nicht in der Whitelist. Überspringe E-Mail.');
                continue; 
            }

            $subject = '';
            if (isset($header->subject)) {
                $mime_header = imap_mime_header_decode($header->subject);
                foreach ($mime_header as $part) {
                    $charset = ($part->charset == 'default') ? 'UTF-8' : $part->charset;
                    $subject .= @iconv($charset, 'UTF-8//TRANSLIT', $part->text);
                }
            }
            $post_title = ( ! empty( $subject ) ) ? sanitize_text_field( $subject ) : 'Beitrag ohne Titel';
            etp_log('Betreff/Titel: ' . $post_title);
            
            $parsed_email = etp_parse_email_body_and_attachments($inbox, $email_number);
            
            $post_content_text = !empty($parsed_email['body_html']) ? $parsed_email['body_html'] : $parsed_email['body_plain'];
            if(empty($post_content_text)) {
                 $post_content_text = !empty($parsed_email['body_plain']) ? nl2br($parsed_email['body_plain']) : '';
            }
            
            // Clean out image file name placeholders from the text.
            $post_content_text = preg_replace('/\[(image|bild|datei|attachment):[^\]]+\]/i', '', $post_content_text);
            etp_log('Inhalt von Bild-Platzhaltern bereinigt.');

            etp_log('Länge des extrahierten Inhalts: ' . strlen($post_content_text) . ' Bytes.');
            
            $attachments = $parsed_email['attachments'];
            etp_log('Habe ' . count($attachments) . ' Anhang/Anhänge gefunden.');

            $attach_id = null;
            
            if ( ! empty( $attachments ) ) {
                $image_attachment = null;
                foreach($attachments as $att) {
                    if (stripos($att['type'], 'image/') === 0) {
                        $image_attachment = $att;
                        etp_log('Bildanhang zur Verarbeitung ausgewählt: ' . $att['filename']);
                        break;
                    }
                }

                if ($image_attachment) {
                    etp_log('Versuche, Bild hochzuladen: ' . $image_attachment['filename']);
                    $upload = wp_upload_bits( $image_attachment['filename'], null, $image_attachment['data'] );
                    etp_log(['Ergebnis von wp_upload_bits:', $upload]);
                    
                    if ( ! $upload['error'] ) {
                        $file_path = $upload['file'];
                        $file_type = wp_check_filetype( basename( $file_path ), null );
                        $attachment = array(
                            'guid'           => $upload['url'], 
                            'post_mime_type' => $file_type['type'],
                            'post_title'     => preg_replace( '/\.[^.]+$/', '', basename( $file_path ) ),
                            'post_content'   => '',
                            'post_status'    => 'inherit'
                        );
                        $attach_id = wp_insert_attachment( $attachment, $file_path );
                        etp_log('Anhang in Mediathek eingefügt mit ID: ' . (is_wp_error($attach_id) ? $attach_id->get_error_message() : $attach_id));

                        if ($attach_id && !is_wp_error($attach_id)) {
                             $attach_data = wp_generate_attachment_metadata( $attach_id, $file_path );
                             wp_update_attachment_metadata( $attach_id, $attach_data );
                             etp_log('Metadaten für Anhang generiert.');
                             etp_log('Bild wird NUR als Beitragsbild gesetzt und NICHT mehr dem Inhalt hinzugefügt.');
                        }
                    } else {
                        etp_log('FEHLER beim Hochladen des Bildes: ' . $upload['error']);
                    }
                } else {
                     etp_log('Kein passender Bildanhang in der Schleife gefunden.');
                }
            }

            $post_data = array(
                'post_title'   => $post_title,
                'post_content' => wp_kses_post( $post_content_text ),
                'post_status'  => 'publish',
                'post_author'  => 1, 
                'post_type'    => 'post',
            );
            $post_id = wp_insert_post( $post_data );
            etp_log('Beitrag erstellt mit ID: ' . (is_wp_error($post_id) ? $post_id->get_error_message() : $post_id));

            if ($post_id && !is_wp_error($post_id)) {
                update_post_meta($post_id, '_etp_generated_post', true);
                etp_log("Post-Meta '_etp_generated_post' für Beitrag {$post_id} gesetzt.");

                if ($attach_id) {
                    set_post_thumbnail($post_id, $attach_id);
                    etp_log("Beitragsbild für Beitrag {$post_id} mit Anhang-ID {$attach_id} gesetzt.");
                }
            }
            
            imap_setflag_full($inbox, $email_number, "\\Seen");
            etp_log("--- Ende der Verarbeitung für E-Mail Nr. {$email_number} ---");
        }
    } else {
        etp_log('Keine neuen E-Mails gefunden.');
    }
    imap_close( $inbox );
    etp_log('Cron-Job beendet.');
}

function etp_get_mime_type($structure) {
    $primary_mime_type = ["TEXT", "MULTIPART", "MESSAGE", "APPLICATION", "AUDIO", "IMAGE", "VIDEO", "OTHER"];
    $mime_type = 'TEXT/PLAIN'; // Default
    if (isset($structure->type)) {
       $mime_type = $primary_mime_type[(int)$structure->type];
       if (isset($structure->subtype)) {
           $mime_type .= "/" . $structure->subtype;
       }
    }
    return strtoupper($mime_type);
}

function etp_parse_email_body_and_attachments($inbox, $email_number) {
    $structure = imap_fetchstructure($inbox, $email_number);
    etp_log(['E-Mail-Struktur:', $structure]);

    $result = [
        'body_html' => '',
        'body_plain' => '',
        'attachments' => []
    ];

    $recursive_parser = function($part_structure, $part_number) use (&$recursive_parser, $inbox, $email_number, &$result) {
        $disposition = (isset($part_structure->disposition)) ? strtolower($part_structure->disposition) : null;
        $mime_type = etp_get_mime_type($part_structure);
        
        $filename = null;
        $params = [];
        if (isset($part_structure->dparameters) && $part_structure->ifdparameters) { $params = array_merge($params, $part_structure->dparameters); }
        if (isset($part_structure->parameters) && $part_structure->ifparameters) { $params = array_merge($params, $part_structure->parameters); }
        foreach ($params as $param) {
            $attribute = strtolower($param->attribute);
            if ($attribute == 'filename' || $attribute == 'name') {
                $decoded_parts = imap_mime_header_decode($param->value);
                $temp_filename = '';
                foreach ($decoded_parts as $p) {
                    $charset = ($p->charset == 'default') ? 'UTF-8' : $p->charset;
                    $temp_filename .= @iconv($charset, 'UTF-8//TRANSLIT', $p->text);
                }
                $filename = $temp_filename;
                break;
            }
        }
        
        if (($disposition == 'attachment' || $disposition == 'inline' || stripos($mime_type, 'image/') === 0) && $filename) {
             etp_log("Verarbeite Anhang/Inline-Teil '{$filename}' in Teil {$part_number}.");
             $data = imap_fetchbody($inbox, $email_number, $part_number);
             switch ($part_structure->encoding) {
                case 3: $data = base64_decode($data); break;
                case 4: $data = quoted_printable_decode($data); break;
             }
             $result['attachments'][] = [ 'filename' => $filename, 'data' => $data, 'type' => $mime_type ];
        } elseif ($mime_type == 'TEXT/HTML') {
            etp_log("Verarbeite HTML-Body in Teil {$part_number}.");
            $data = imap_fetchbody($inbox, $email_number, $part_number);
            switch ($part_structure->encoding) {
                case 3: $data = base64_decode($data); break;
                case 4: $data = quoted_printable_decode($data); break;
            }
            if(strlen($data) > strlen($result['body_html'])) {
                $result['body_html'] = $data;
            }
        } elseif ($mime_type == 'TEXT/PLAIN') {
             etp_log("Verarbeite Plain-Text-Body in Teil {$part_number}.");
             $data = imap_fetchbody($inbox, $email_number, $part_number);
             switch ($part_structure->encoding) {
                case 3: $data = base64_decode($data); break;
                case 4: $data = quoted_printable_decode($data); break;
             }
             if(strlen($data) > strlen($result['body_plain'])) {
                $result['body_plain'] = $data;
             }
        }

        if (isset($part_structure->parts)) {
            foreach ($part_structure->parts as $index => $sub_part) {
                $sub_part_number = ($part_number ? $part_number . '.' : '') . ($index + 1);
                $recursive_parser($sub_part, $sub_part_number);
            }
        }
    };
    
    // Initial call for multipart emails
    if (isset($structure->parts)) {
        foreach ($structure->parts as $index => $part) {
            $recursive_parser($part, $index + 1);
        }
    } else { // Handle non-multipart emails
        etp_log("E-Mail ist nicht multipart, versuche Hauptteil zu lesen.");
        $recursive_parser($structure, '1');
    }

    return $result;
}


// ==============
// FRONTEND-DARSTELLUNG
// ==============

// Fügt eine benutzerdefinierte CSS-Klasse zum <body>-Tag hinzu
function etp_add_body_class($classes) {
    $settings = get_option('etp_settings');
    $is_design_enabled = isset($settings['enable_modern_design']) ? $settings['enable_modern_design'] : true;

    if ( $is_design_enabled && is_single() && get_the_ID() && get_post_meta(get_the_ID(), '_etp_generated_post', true)) {
        $classes[] = 'etp-body-styling';
    }
    return $classes;
}
add_filter('body_class', 'etp_add_body_class');

// Fügt eine benutzerdefinierte CSS-Klasse zu den Beiträgen hinzu, die vom Plugin erstellt wurden
function etp_add_post_class($classes) {
    if (is_single() && get_the_ID() && get_post_meta(get_the_ID(), '_etp_generated_post', true)) {
        $classes[] = 'etp-styled-post';
    }
    return $classes;
}
add_filter('post_class', 'etp_add_post_class');

// Fügt das benutzerdefinierte CSS für die Beitragsdarstellung hinzu
function etp_enqueue_styles() {
    $settings = get_option('etp_settings');
    $is_design_enabled = isset($settings['enable_modern_design']) ? $settings['enable_modern_design'] : true;

    if ( $is_design_enabled && is_single() && get_the_ID() && get_post_meta(get_the_ID(), '_etp_generated_post', true)) {
        $custom_css = "
            /* Globaler Stil für die Seite mit einem ETP-Beitrag */
            .etp-body-styling {
                background-color: #f4f5f7;
            }
            @keyframes etp-fade-in-up {
                from { opacity: 0; transform: translateY(30px); }
                to { opacity: 1; transform: translateY(0); }
            }

            /* Hauptcontainer des Beitrags */
            .etp-styled-post {
                max-width: 760px;
                margin: 60px auto !important;
                background: #ffffff;
                border-radius: 8px;
                box-shadow: 0 4px 6px rgba(0,0,0,0.04), 0 10px 30px rgba(0,0,0,0.08);
                overflow: hidden;
                border: none;
                padding: 0 !important;
                animation: etp-fade-in-up 0.7s cubic-bezier(0.25, 0.8, 0.25, 1);
                animation-fill-mode: forwards;
                opacity: 0;
            }

            /* Beitragsbild-Container */
            .etp-styled-post .post-thumbnail {
                margin: 0;
                padding: 0;
                line-height: 0;
                overflow: hidden;
            }
            .etp-styled-post .post-thumbnail img {
                width: 100%;
                height: 400px;
                object-fit: cover;
                border-radius: 0;
                margin-bottom: 0;
                transition: transform 0.5s cubic-bezier(0.25, 0.8, 0.25, 1);
                animation: inherit;
                animation-delay: 0.1s;
                opacity: 0;
            }
            .etp-styled-post:hover .post-thumbnail img {
                transform: scale(1.05);
            }
            
            /* Inhalts-Wrapper */
            .etp-styled-post .entry-header,
            .etp-styled-post .entry-content {
                padding: 40px 50px;
            }
            .etp-styled-post .entry-header {
                padding-bottom: 20px;
                animation: inherit;
                animation-delay: 0.2s;
                opacity: 0;
            }
            .etp-styled-post .entry-content {
                padding-top: 0;
                animation: inherit;
                animation-delay: 0.3s;
                opacity: 0;
            }

            /* Titel-Styling */
            .etp-styled-post .entry-title {
                font-family: Georgia, 'Times New Roman', Times, serif;
                font-size: 2.8em;
                font-weight: 700;
                color: #1a1a1a;
                margin: 0 0 0.5em 0;
                line-height: 1.15;
            }
            
            /* Inhalts-Typografie */
            .etp-styled-post .entry-content {
                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Helvetica, Arial, sans-serif, 'Apple Color Emoji', 'Segoe UI Emoji', 'Segoe UI Symbol';
                font-size: 1.15rem;
                line-height: 1.8;
                color: #333;
            }
            .etp-styled-post .entry-content p {
                margin-bottom: 1.5em;
            }

            .etp-styled-post .entry-content a {
                color: #6d5cae;
                text-decoration: none;
                background-image: linear-gradient(120deg, #e0c3fc 0%, #8ec5fc 100%);
                background-repeat: no-repeat;
                background-size: 100% 0.2em;
                background-position: 0 88%;
                transition: background-size 0.25s ease-in;
            }
            .etp-styled-post .entry-content a:hover {
                background-size: 100% 88%;
            }

             .etp-styled-post .entry-content blockquote {
                border-left: 3px solid #6d5cae;
                padding-left: 20px;
                margin-left: 0;
                font-style: italic;
                color: #666;
            }
        ";
        wp_register_style('etp-inline-styles', false);
        wp_enqueue_style('etp-inline-styles');
        wp_add_inline_style('etp-inline-styles', $custom_css);
    }
}
add_action('wp_enqueue_scripts', 'etp_enqueue_styles');


// ==============
// ADMIN-MENÜ & EINSTELLUNGSSEITE
// ==============

function etp_add_admin_menu() {
    add_submenu_page('edit.php', 'E-Mail zu Beitrag Einstellungen', 'E-Mail zu Beitrag', 'manage_options', 'email-to-post-settings', 'etp_settings_page_callback');
}
add_action( 'admin_menu', 'etp_add_admin_menu' );

// AJAX-Handler für den Verbindungstest
function etp_ajax_test_connection() {
    check_ajax_referer('etp_test_connection_nonce', 'nonce');
    if (!function_exists('imap_open')) { wp_send_json_error(['message' => 'Die PHP IMAP-Erweiterung ist auf Ihrem Server nicht aktiviert.']); return; }

    $host = sanitize_text_field($_POST['host']);
    $port = sanitize_text_field($_POST['port']);
    $user = sanitize_email($_POST['user']);
    $pass = stripslashes($_POST['pass']);
    $encryption = sanitize_text_field($_POST['encryption']);
    $connection_string = '{' . $host . ':' . $port . '/imap/' . $encryption . '/novalidate-cert}INBOX';
    
    $error_message = '';
    set_error_handler(function($errno, $errstr) use (&$error_message) { $error_message = $errstr; });
    $inbox = imap_open($connection_string, $user, $pass, OP_READONLY, 1, ['DISABLE_AUTHENTICATOR' => 'GSSAPI']);
    restore_error_handler();

    if ($inbox) {
        imap_close($inbox);
        wp_send_json_success(['message' => 'Verbindung erfolgreich hergestellt!']);
    } else {
        $simple_message = 'Verbindung fehlgeschlagen';
        $help_text = '';

        if (stripos($error_message, 'authentication failed') !== false) {
            $simple_message = 'Authentifizierung fehlgeschlagen';
            $help_text = '<h4>Was bedeutet das?</h4><p>Ihr Server konnte sich nicht am E-Mail-Postfach anmelden. Das bedeutet fast immer, dass Benutzername oder Passwort falsch sind.</p><h4>Mögliche Lösungen:</h4><ul><li><strong>Passwort überprüfen:</strong> Geben Sie Ihr Passwort erneut ein. Achten Sie auf Tippfehler und die Feststelltaste.</li><li><strong>App-Passwort verwenden (Wichtig!):</strong> Viele Anbieter (wie Gmail, GMX, Outlook) verlangen für die Verbindung über externe Apps (wie dieses Plugin) ein separates "App-Passwort". Ihr normales Login-Passwort funktioniert hier nicht.<ul><li><a href="https://support.google.com/accounts/answer/185833" target="_blank">Anleitung für Gmail / Google Workspace</a></li><li><a href="https://support.microsoft.com/de-de/account-billing/verwalten-von-app-kennw-rtern-f-r-die-verifizierung-in-zwei-schritten-d6dc8c6e-4bf9-4851-8744-67e453b7b2c9" target="_blank">Anleitung für Outlook / Microsoft 365</a></li><li><a href="https://hilfe.gmx.net/sicherheit/app-passwort.html" target="_blank">Anleitung für GMX</a></li></ul></li><li><strong>Benutzername überprüfen:</strong> Stellen Sie sicher, dass Sie die vollständige E-Mail-Adresse als Benutzernamen verwenden.</li></ul>';
        } elseif (stripos($error_message, 'timed out') !== false || stripos($error_message, 'couldn\'t connect') !== false) {
            $simple_message = 'Server nicht erreichbar';
            $help_text = '<h4>Was bedeutet das?</h4><p>Ihr WordPress-Server konnte keine Verbindung zum E-Mail-Server unter der angegebenen Adresse (Host) und dem Port aufbauen. Der Server antwortet nicht.</p><h4>Mögliche Lösungen:</h4><ul><li><strong>Host-Namen überprüfen:</strong> Kontrollieren Sie den IMAP-Host auf Tippfehler. Korrekte Adressen sind z.B. <code>imap.gmail.com</code> oder <code>mail.ihredomain.de</code>.</li><li><strong>Port überprüfen:</strong> Die Standard-Ports für eine sichere Verbindung sind <code>993</code> (für SSL) und seltener <code>143</code> (für TLS/STARTTLS). Stellen Sie sicher, dass Sie den richtigen Port für Ihre Verschlüsselungsmethode verwenden.</li><li><strong>Firewall:</strong> In seltenen Fällen blockiert die Firewall Ihres Webservers ausgehende Verbindungen auf diesem Port. Kontaktieren Sie Ihren Hoster und fragen Sie, ob ausgehende Verbindungen zu <code>' . esc_html($host) . '</code> auf Port <code>' . esc_html($port) . '</code> erlaubt sind.</li></ul>';
        } elseif (stripos($error_message, 'certificate') !== false) {
            $simple_message = 'Zertifikatsfehler';
            $help_text = '<h4>Was bedeutet das?</h4><p>Der E-Mail-Server verwendet ein SSL-Zertifikat, dem Ihr WordPress-Server nicht vertraut. Dies kann bei manchen Hostern vorkommen.</p><h4>Lösung:</h4><p>Das Plugin versucht bereits, dieses Problem automatisch zu umgehen (mit <code>/novalidate-cert</code>). Wenn der Fehler trotzdem auftritt, liegt wahrscheinlich ein tiefergreifendes Konfigurationsproblem auf dem Server vor. Kontaktieren Sie Ihren Hoster mit der ursprünglichen Fehlermeldung: <code>' . esc_html($error_message) . '</code></p>';
        } else {
            $simple_message = 'Unbekannter Verbindungsfehler';
            $help_text = '<h4>Was ist passiert?</h4><p>Es ist ein unerwarteter Fehler aufgetreten. Bitte überprüfen Sie alle Zugangsdaten und Einstellungen sorgfältig.</p><h4>Ursprüngliche Fehlermeldung (für Experten oder den Hoster-Support):</h4><p><code>' . esc_html(imap_last_error()) . '</code></p>';
        }
        wp_send_json_error(['message' => $simple_message, 'help' => $help_text]);
    }
}
add_action('wp_ajax_etp_test_connection', 'etp_ajax_test_connection');

// AJAX-Handler zum Löschen der Log-Datei
function etp_ajax_clear_log() {
    check_ajax_referer('etp_clear_log_nonce', 'nonce');
    $upload_dir = wp_upload_dir();
    $log_file = $upload_dir['basedir'] . '/email-to-post-debug.log';
    if (file_exists($log_file)) {
        unlink($log_file);
        wp_send_json_success();
    } else {
        wp_send_json_error();
    }
}
add_action('wp_ajax_etp_clear_log', 'etp_ajax_clear_log');


function etp_settings_page_callback() {
    if ( isset( $_POST['etp_settings_nonce'] ) && wp_verify_nonce( $_POST['etp_settings_nonce'], 'etp_save_settings' ) ) {
        $current_settings = get_option('etp_settings', []);
        $current_settings['imap_host'] = isset($_POST['etp_imap_host']) ? sanitize_text_field($_POST['etp_imap_host']) : '';
        $current_settings['imap_port'] = isset($_POST['etp_imap_port']) ? sanitize_text_field($_POST['etp_imap_port']) : '993';
        $current_settings['imap_user'] = isset($_POST['etp_imap_user']) ? sanitize_email($_POST['etp_imap_user']) : '';
        $current_settings['imap_encryption'] = isset($_POST['etp_imap_encryption']) ? sanitize_text_field($_POST['etp_imap_encryption']) : 'ssl';
        if ( ! empty( $_POST['etp_imap_pass'] ) ) { $current_settings['imap_pass'] = stripslashes($_POST['etp_imap_pass']); }
        $current_settings['debug_mode'] = isset($_POST['etp_debug_mode']);
        $current_settings['enable_modern_design'] = isset($_POST['etp_enable_modern_design']);

        $allowed_emails_raw = isset($_POST['etp_allowed_emails']) ? $_POST['etp_allowed_emails'] : [];
        $allowed_emails_clean = [];
        foreach ($allowed_emails_raw as $email) { if (is_email($email)) { $allowed_emails_clean[] = sanitize_email($email); } }
        $current_settings['allowed_emails'] = array_unique($allowed_emails_clean);
        update_option('etp_settings', $current_settings);
        echo '<div id="etp-save-notice" class="notice notice-success is-dismissible"><p>Einstellungen erfolgreich gespeichert!</p></div>';
    }
    $settings = get_option('etp_settings');
    $is_imap_enabled = function_exists('imap_open');
    $upload_dir = wp_upload_dir();
    $is_uploads_writable = is_writable($upload_dir['basedir']);
    ?>
    <style>
        :root { --etp-primary-color: #6d5cae; --etp-secondary-color: #8874c2; --etp-bg-color: #f9f9fd; --etp-card-bg: #ffffff; --etp-text-color: #444; --etp-border-color: #e9e9f0; --etp-success-color: #28a745; --etp-error-color: #dc3545; --etp-status-ok-bg: #eaf6ec; --etp-status-error-bg: #fbe9ea; --etp-info-bg: #e9f5fe; --etp-info-color: #0369a1; }
        .etp-admin-wrap { background-color: var(--etp-bg-color); color: var(--etp-text-color); padding: 20px; margin: 20px auto; max-width: 960px; border-radius: 12px; animation: etp-fade-in 0.5s ease-out; }
        @keyframes etp-fade-in { from { opacity: 0; transform: translateY(15px); } to { opacity: 1; transform: translateY(0); } }
        .etp-header { background: linear-gradient(135deg, var(--etp-secondary-color), var(--etp-primary-color)); color: white; padding: 30px; border-radius: 10px; text-align: center; box-shadow: 0 8px 25px rgba(109, 92, 174, 0.2); }
        .etp-header h1 { color: white; margin: 0 0 10px 0; font-size: 28px; }
        .etp-tabs { display: flex; flex-wrap: wrap; justify-content: center; border-bottom: 2px solid var(--etp-border-color); margin: 30px 0; }
        .etp-tab-button { background: none; border: none; padding: 15px 25px; cursor: pointer; color: #777; font-weight: 600; position: relative; transition: color 0.3s; }
        .etp-tab-button::after { content: ''; position: absolute; bottom: -2px; left: 0; right: 0; height: 2px; background-color: var(--etp-primary-color); transform: scaleX(0); transition: transform 0.3s ease; }
        .etp-tab-button.active { color: var(--etp-primary-color); }
        .etp-tab-button.active::after { transform: scaleX(1); }
        .etp-tab-content { display: none; }
        .etp-tab-content.active { display: block; animation: etp-fade-in 0.4s; }
        .etp-card { background: var(--etp-card-bg); border: 1px solid var(--etp-border-color); border-radius: 10px; padding: 30px; margin-bottom: 20px; box-shadow: 0 5px 20px rgba(0,0,0,0.05); }
        .etp-card h2 { margin-top: 0; }
        .etp-form-grid { display: grid; grid-template-columns: 200px 1fr; gap: 15px 20px; align-items: center; }
        .etp-form-grid label { font-weight: 600; }
        .etp-form-grid input, .etp-form-grid select { width: 100%; max-width: 400px; }
        .etp-email-list { display: flex; flex-direction: column; gap: 10px; }
        .etp-email-item { display: flex; align-items: center; gap: 10px; }
        .etp-email-item input { flex-grow: 1; }
        .etp-remove-email { color: #aaa; font-size: 24px; text-decoration: none; transition: color 0.2s; }
        .etp-remove-email:hover { color: var(--etp-error-color); }
        #etp-test-result { margin-top: 15px; padding: 10px; border-radius: 5px; font-weight: 600; display: none; }
        #etp-test-result.success { background-color: var(--etp-status-ok-bg); color: var(--etp-success-color); }
        #etp-test-result.error { background-color: var(--etp-status-error-bg); color: var(--etp-error-color); }
        #etp-test-help { margin-top: 10px; padding: 15px 20px; border-radius: 5px; background-color: var(--etp-info-bg); color: var(--etp-info-color); display: none; border-left: 4px solid var(--etp-info-color); }
        #etp-test-help h4 { margin: 0 0 10px 0; color: #025277; }
        #etp-test-help ul { margin: 0; padding-left: 20px; }
        #etp-test-help a { color: #025277; font-weight: bold; }
        .etp-status-box { padding: 15px; border-radius: 8px; margin-bottom: 20px; display: flex; align-items: center; gap: 15px; }
        .etp-status-box.ok { background-color: var(--etp-status-ok-bg); color: var(--etp-success-color); }
        .etp-status-box.error { background-color: var(--etp-status-error-bg); color: var(--etp-error-color); }
        .etp-status-box .status-icon { font-size: 24px; }
        .etp-status-box .status-text strong { display: block; }
        #support-request-template, #etp_debug_log { width: 100%; min-height: 200px; font-family: monospace; font-size: 12px; background: #f3f4f6; border: 1px solid #e5e7eb; }
        .etp-admin-wrap .notice-success { background-color: var(--etp-status-ok-bg) !important; border-left-color: var(--etp-success-color) !important; color: var(--etp-success-color) !important; }
        .etp-tutorial .step { margin-bottom: 30px; }
        .etp-tutorial .step h3 { display: flex; align-items: center; gap: 12px; color: var(--etp-primary-color); }
        .etp-tutorial .step h3 .num { background-color: var(--etp-primary-color); color: white; border-radius: 50%; width: 30px; height: 30px; display: inline-flex; justify-content: center; align-items: center; font-size: 16px; }
        .etp-tutorial .settings-example { background-color: #f3f4f6; border: 1px solid #e5e7eb; padding: 15px; border-radius: 6px; font-family: monospace; }
        .etp-tutorial .faq details { margin-bottom: 10px; background: #fafafa; border: 1px solid var(--etp-border-color); border-radius: 6px; }
        .etp-tutorial .faq summary { padding: 15px; font-weight: bold; cursor: pointer; }
        .etp-tutorial .faq div { padding: 0 15px 15px; }
    </style>

    <div class="wrap etp-admin-wrap">
        <div class="etp-header">
            <h1><?php esc_html_e( 'E-Mail zu Beitrag: Einstellungen', 'email-to-post' ); ?></h1>
            <p>Konfigurieren Sie Ihr E-Mail-Postfach, um Beiträge automatisch zu erstellen.</p>
        </div>

        <nav class="etp-tabs">
            <button class="etp-tab-button" data-tab="tab-check">System-Check</button>
            <button class="etp-tab-button" data-tab="tab-settings">Server-Einstellungen</button>
            <button class="etp-tab-button" data-tab="tab-whitelist">Erlaubte Absender</button>
            <button class="etp-tab-button" data-tab="tab-tutorial">Anleitung</button>
            <button class="etp-tab-button" data-tab="tab-design">Darstellung</button>
            <button class="etp-tab-button" data-tab="tab-debug">Debug</button>
        </nav>

        <form method="post" action="">
            <?php wp_nonce_field( 'etp_save_settings', 'etp_settings_nonce' ); ?>
            
            <div id="tab-check" class="etp-tab-content etp-card">
                <h2>Systemvoraussetzungen-Check</h2>
                <div class="etp-status-box <?php echo $is_imap_enabled ? 'ok' : 'error'; ?>">
                    <span class="status-icon"><?php echo $is_imap_enabled ? '✔' : '✖'; ?></span>
                    <div class="status-text">
                        <strong>PHP IMAP-Erweiterung</strong>
                        <span>Status: <?php echo $is_imap_enabled ? 'Aktiv & betriebsbereit' : 'Nicht gefunden'; ?></span>
                    </div>
                </div>
                 <div class="etp-status-box <?php echo $is_uploads_writable ? 'ok' : 'error'; ?>">
                    <span class="status-icon"><?php echo $is_uploads_writable ? '✔' : '✖'; ?></span>
                    <div class="status-text">
                        <strong>Uploads-Verzeichnis beschreibbar</strong>
                        <span>Status: <?php echo $is_uploads_writable ? 'Beschreibbar' : 'Nicht beschreibbar'; ?></span>
                    </div>
                </div>
                <?php if (!$is_imap_enabled): ?>
                    <h3>Nächster Schritt</h3>
                    <p>Damit dieses Plugin funktionieren kann, muss Ihr Hoster eine Standard-PHP-Erweiterung aktivieren. Das ist in der Regel kostenlos und schnell erledigt.</p>
                    <p><strong>Kopieren Sie einfach den folgenden Text und senden Sie ihn an den Support Ihres Webhosters:</strong></p>
                    <textarea id="support-request-template" readonly>
Sehr geehrtes Support-Team,

ich möchte auf meiner WordPress-Installation das Plugin "E-Mail zu Beitrag" verwenden. Dafür ist die Aktivierung der PHP-Erweiterung "IMAP" (php-imap) auf meinem Server notwendig.

Könnten Sie diese Erweiterung bitte für meine Domain <?php echo esc_html(site_url()); ?> aktivieren?

Vielen Dank und freundliche Grüße,
[Ihr Name]
                    </textarea>
                    <button type="button" class="button" onclick="document.getElementById('support-request-template').select(); document.execCommand('copy');">Text kopieren</button>
                <?php endif; ?>
                 <?php if (!$is_uploads_writable): ?>
                    <h3>Aktion erforderlich: Schreibrechte anpassen</h3>
                    <p>Das Plugin kann keine Bilder hochladen, weil Ihr Uploads-Verzeichnis (<code>wp-content/uploads</code>) nicht für den Webserver beschreibbar ist.</p>
                    <p><strong>Lösung:</strong> Verbinden Sie sich per FTP mit Ihrem Server und setzen Sie die Berechtigungen für den Ordner <code>wp-content/uploads</code> auf <code>755</code>. Falls Sie unsicher sind, wie das geht, kontaktieren Sie bitte den Support Ihres Hosters.</p>
                <?php endif; ?>
            </div>

            <div id="tab-settings" class="etp-tab-content etp-card">
                <h2>IMAP-Verbindung</h2>
                <div class="etp-form-grid">
                    <label for="etp_imap_host">IMAP Host</label>
                    <input name="etp_imap_host" type="text" id="etp_imap_host" value="<?php echo esc_attr( $settings['imap_host'] ?? '' ); ?>" placeholder="imap.beispiel.de" <?php disabled(!$is_imap_enabled); ?>>
                    <label for="etp_imap_port">IMAP Port</label>
                    <input name="etp_imap_port" type="number" id="etp_imap_port" value="<?php echo esc_attr( $settings['imap_port'] ?? '993' ); ?>" <?php disabled(!$is_imap_enabled); ?>>
                    <label for="etp_imap_encryption">Verschlüsselung</label>
                    <select name="etp_imap_encryption" id="etp_imap_encryption" <?php disabled(!$is_imap_enabled); ?>>
                        <option value="ssl" <?php selected( ($settings['imap_encryption'] ?? 'ssl'), 'ssl' ); ?>>SSL (Port 993)</option>
                        <option value="tls" <?php selected( ($settings['imap_encryption'] ?? ''), 'tls' ); ?>>TLS (Port 143)</option>
                        <option value="notls" <?php selected( ($settings['imap_encryption'] ?? ''), 'notls' ); ?>>Keine</option>
                    </select>
                    <label for="etp_imap_user">Benutzername</label>
                    <input name="etp_imap_user" type="email" id="etp_imap_user" value="<?php echo esc_attr( $settings['imap_user'] ?? '' ); ?>" placeholder="ihre-adresse@beispiel.de" <?php disabled(!$is_imap_enabled); ?>>
                    <label for="etp_imap_pass">Passwort</label>
                    <input name="etp_imap_pass" type="password" id="etp_imap_pass" value="" placeholder="•••••••••••" <?php disabled(!$is_imap_enabled); ?>>
                    <span></span><p class="description">Lassen Sie dieses Feld leer, um das gespeicherte Passwort nicht zu ändern.</p>
                </div>
                <div style="margin-top: 20px;">
                    <button type="button" class="button button-primary" id="etp-test-connection-btn" <?php disabled(!$is_imap_enabled); ?>>Verbindung testen</button>
                    <span class="spinner" style="float: none; vertical-align: middle;"></span>
                    <div id="etp-test-result"></div>
                    <div id="etp-test-help"></div>
                </div>
            </div>

            <div id="tab-whitelist" class="etp-tab-content etp-card">
                 <h2>Whitelist der Absender</h2>
                <p>Nur E-Mails von diesen Adressen werden in Beiträge umgewandelt.</p>
                <div id="etp_email_list_container" class="etp-email-list">
                    <?php 
                    $allowed_emails = $settings['allowed_emails'] ?? [];
                    if ( ! empty($allowed_emails) ) :
                        foreach ($allowed_emails as $email) : ?>
                            <div class="etp-email-item">
                                <input type="email" name="etp_allowed_emails[]" value="<?php echo esc_attr($email); ?>" required <?php disabled(!$is_imap_enabled); ?>>
                                <a href="#" class="etp-remove-email" title="Entfernen">&times;</a>
                            </div>
                        <?php endforeach;
                    else: ?>
                         <p>Noch keine E-Mail-Adressen hinzugefügt.</p>
                    <?php endif; ?>
                </div>
                <button type="button" class="button etp-add-email" style="margin-top: 15px;" <?php disabled(!$is_imap_enabled); ?>>+ E-Mail-Adresse hinzufügen</button>
            </div>

             <div id="tab-tutorial" class="etp-tab-content etp-card etp-tutorial">
                <h2>So funktioniert's: Von der E-Mail zum Beitrag</h2>
                <div class="step">
                    <h3><span class="num">1</span> Ein separates E-Mail-Postfach einrichten</h3>
                    <p>Der wichtigste Schritt für einen reibungslosen Ablauf ist ein <strong>dediziertes E-Mail-Postfach</strong>. Nutzen Sie nicht Ihr persönliches oder geschäftliches Hauptpostfach!</p>
                    <ul>
                        <li><strong>Warum?</strong> Das Plugin liest alle ungelesenen E-Mails im Posteingang. Ein separates Postfach (z.B. <code>news@ihredomain.de</code>) verhindert, dass versehentlich private E-Mails als Beiträge veröffentlicht werden.</li>
                        <li><strong>Was wird benötigt?</strong> Sie benötigen die IMAP-Zugangsdaten für dieses Postfach. Diese finden Sie in der Regel in der Dokumentation oder im Kundenbereich Ihres E-Mail-Anbieters/Hosters.</li>
                    </ul>
                </div>

                <div class="step">
                    <h3><span class="num">2</span> Plugin konfigurieren & Verbindung testen</h3>
                    <p>Tragen Sie die IMAP-Daten Ihres neuen Postfachs im Tab "Server-Einstellungen" ein.</p>
                    <div class="settings-example">
                        <strong>Beispiel für einen gängigen Anbieter:</strong><br>
                        IMAP Host: <code>imap.anbieter.de</code><br>
                        IMAP Port: <code>993</code><br>
                        Verschlüsselung: <code>SSL (Port 993)</code><br>
                        Benutzername: <code>news@ihredomain.de</code><br>
                        Passwort: <code>Ihr_App-Passwort_oder_ normales_Passwort</code>
                    </div>
                    <p>Klicken Sie danach unbedingt auf <strong>"Verbindung testen"</strong>. Das Plugin gibt Ihnen eine klare Rückmeldung, ob die Verbindung klappt und bietet bei Problemen detaillierte Lösungshilfen an.</p>
                </div>
                
                <div class="step">
                    <h3><span class="num">3</span> Die perfekte E-Mail verfassen</h3>
                    <p>Das Plugin wandelt die Elemente Ihrer E-Mail wie folgt um:</p>
                    <ul>
                        <li><strong>Empfänger:</strong> Die eingerichtete E-Mail-Adresse (z.B. <code>news@ihredomain.de</code>).</li>
                        <li><strong>Absender:</strong> Muss im Tab "Erlaubte Absender" freigegeben sein.</li>
                        <li><strong>Betreff:</strong> Wird zum <strong>Titel</strong> des WordPress-Beitrags.</li>
                        <li><strong>E-Mail-Text:</strong> Wird zum <strong>Inhalt</strong> des Beitrags. Formatierungen wie <strong>fett</strong>, <em>kursiv</em>, <a href="#">Links</a> und Absätze werden übernommen.</li>
                        <li><strong>Anhänge:</strong> Das <strong>erste angehängte Bild</strong> (z.B. .jpg, .png) wird in die Mediathek hochgeladen und als <strong>Beitragsbild</strong> gesetzt. Es erscheint NICHT mehr im Text.</li>
                    </ul>
                </div>
                
                <div class="faq">
                    <h3>Häufige Fragen (FAQ)</h3>
                    <details>
                        <summary>Was ist ein "App-Passwort" und warum brauche ich das?</summary>
                        <div>
                            <p>Anbieter wie Google, Microsoft oder GMX haben hohe Sicherheitsstandards. Wenn Sie sich mit einer externen Anwendung (wie diesem Plugin) anmelden wollen, reicht Ihr normales Passwort oft nicht aus. Sie müssen in den Sicherheitseinstellungen Ihres E-Mail-Kontos ein spezielles Passwort nur für diese eine Anwendung erstellen. Das erhöht die Sicherheit, da Sie dieses Passwort jederzeit widerrufen können, ohne Ihr Hauptpasswort ändern zu müssen.</p>
                        </div>
                    </details>
                    <details>
                        <summary>Meine E-Mail wurde gesendet, aber es erscheint kein Beitrag. Woran liegt das?</summary>
                        <div>
                            <p>Überprüfen Sie diese Punkte:</p>
                            <ol>
                                <li><strong>Ist der Absender freigegeben?</strong> Schauen Sie im Tab "Erlaubte Absender" nach, ob die E-Mail-Adresse, von der Sie gesendet haben, exakt so eingetragen ist.</li>
                                <li><strong>WordPress Cron-Job:</strong> Das Plugin prüft alle 5 Minuten auf neue E-Mails. Bitte haben Sie einen Moment Geduld. Falls es dauerhaft nicht funktioniert, könnte der WordPress-Cron auf Ihrem Server deaktiviert oder fehlerhaft sein. Kontaktieren Sie hierzu Ihren Hoster.</li>
                                <li><strong>E-Mail im Posteingang?</strong> Stellen Sie sicher, dass die E-Mail auch wirklich im Posteingang (INBOX) des Ziel-Postfachs gelandet ist und nicht in einem Spam-Ordner.</li>
                            </ol>
                        </div>
                    </details>
                </div>
            </div>

            <div id="tab-design" class="etp-tab-content etp-card">
                <h2>Darstellung der Beiträge</h2>
                <p>Hier können Sie das Aussehen der per E-Mail erstellten Beiträge anpassen.</p>
                <label style="font-weight: bold; display: block; margin-bottom: 10px;">
                    <input type="checkbox" name="etp_enable_modern_design" <?php checked(isset($settings['enable_modern_design']) ? $settings['enable_modern_design'] : true ); ?>>
                    Modernes Beitrags-Design aktivieren
                </label>
                <p class="description">Wenn diese Option aktiviert ist, wendet das Plugin ein eigenes, modernes Layout mit Animationen auf die erstellten Beiträge an. Deaktivieren Sie diese Option, wenn Ihr WordPress-Theme die Darstellung vollständig übernehmen soll.</p>
            </div>

            <div id="tab-debug" class="etp-tab-content etp-card">
                <h2>Debug & Diagnose</h2>
                <p>Wenn Beiträge nicht wie erwartet erstellt werden, hilft Ihnen dieser Bereich bei der Fehlersuche.</p>
                <label style="font-weight: bold; display: block; margin-bottom: 10px;">
                    <input type="checkbox" name="etp_debug_mode" <?php checked(isset($settings['debug_mode']) && $settings['debug_mode']); ?>>
                    Debug-Modus aktivieren
                </label>
                <p class="description">Wenn aktiviert, schreibt das Plugin ein detailliertes Protokoll aller Aktionen in die Datei <code>wp-content/uploads/email-to-post-debug.log</code>. <strong>Achtung:</strong> Deaktivieren Sie diese Option nach der Fehlersuche wieder, da die Log-Datei sonst sehr groß werden kann.</p>
                
                <h3>Protokoll-Ausgabe</h3>
                <textarea id="etp_debug_log" readonly><?php 
                    $log_file = $upload_dir['basedir'] . '/email-to-post-debug.log';
                    if (file_exists($log_file)) {
                        echo esc_textarea(file_get_contents($log_file));
                    } else {
                        echo 'Noch kein Protokoll vorhanden. Aktivieren Sie den Debug-Modus und warten Sie auf den nächsten E-Mail-Abruf.';
                    }
                ?></textarea>
                <button type="button" class="button" id="etp-clear-log-btn" style="margin-top: 10px;">Protokoll löschen</button>
            </div>

            <?php if ($is_imap_enabled) submit_button('Einstellungen speichern'); ?>
        </form>
    </div>

    <script>
    document.addEventListener('DOMContentLoaded', function() {
        const isImapEnabled = <?php echo json_encode($is_imap_enabled); ?>;
        const defaultTab = isImapEnabled ? 'tab-settings' : 'tab-check';
        document.querySelectorAll('.etp-tab-button').forEach(btn => btn.classList.remove('active'));
        document.querySelectorAll('.etp-tab-content').forEach(content => content.classList.remove('active'));
        document.querySelector(`[data-tab="${defaultTab}"]`).classList.add('active');
        document.getElementById(defaultTab).classList.add('active');

        document.querySelectorAll('.etp-tab-button').forEach(button => {
            button.addEventListener('click', function(e) {
                e.preventDefault();
                document.querySelectorAll('.etp-tab-button').forEach(btn => btn.classList.remove('active'));
                document.querySelectorAll('.etp-tab-content').forEach(content => content.classList.remove('active'));
                this.classList.add('active');
                document.getElementById(this.dataset.tab).classList.add('active');
                window.location.hash = this.dataset.tab;
            });
        });

        if (window.location.hash) {
            const tabButton = document.querySelector(`[data-tab="${window.location.hash.substring(1)}"]`);
            if (tabButton) { tabButton.click(); }
        }

        const emailListContainer = document.getElementById('etp_email_list_container');
        if (emailListContainer) {
            document.querySelector('.etp-add-email').addEventListener('click', function() {
                const p = emailListContainer.querySelector('p');
                if (p) p.remove();
                const newItem = document.createElement('div');
                newItem.className = 'etp-email-item';
                newItem.innerHTML = `<input type="email" name="etp_allowed_emails[]" placeholder="neue-adresse@beispiel.de" required><a href="#" class="etp-remove-email" title="Entfernen">&times;</a>`;
                emailListContainer.appendChild(newItem);
            });
            emailListContainer.addEventListener('click', function(e) {
                if (e.target.classList.contains('etp-remove-email')) {
                    e.preventDefault();
                    e.target.parentElement.remove();
                    if(emailListContainer.children.length === 0) {
                         const p = document.createElement('p');
                         p.textContent = 'Noch keine E-Mail-Adressen hinzugefügt.';
                         emailListContainer.appendChild(p);
                    }
                }
            });
        }
        
        const clearLogBtn = document.getElementById('etp-clear-log-btn');
        if(clearLogBtn) {
            clearLogBtn.addEventListener('click', async function(e){
                e.preventDefault();
                // Removed the confirm() dialog to comply with requirements.
                const formData = new FormData();
                formData.append('action', 'etp_clear_log');
                formData.append('nonce', '<?php echo wp_create_nonce("etp_clear_log_nonce"); ?>');
                try {
                     await fetch('<?php echo admin_url('admin-ajax.php'); ?>', { method: 'POST', body: formData });
                     document.getElementById('etp_debug_log').value = 'Protokoll wurde gelöscht.';
                } catch(error) {
                    // Using console.error instead of alert
                    console.error('Fehler beim Löschen des Protokolls.', error);
                }
            });
        }

        const testBtn = document.getElementById('etp-test-connection-btn');
        if (testBtn) {
            const spinner = testBtn.nextElementSibling;
            const resultDiv = document.getElementById('etp-test-result');
            const helpDiv = document.getElementById('etp-test-help');
            testBtn.addEventListener('click', async function() {
                spinner.style.visibility = 'visible';
                resultDiv.style.display = 'none';
                helpDiv.style.display = 'none';
                resultDiv.className = '';
                helpDiv.innerHTML = '';

                const formData = new FormData();
                formData.append('action', 'etp_test_connection');
                formData.append('nonce', '<?php echo wp_create_nonce("etp_test_connection_nonce"); ?>');
                formData.append('host', document.getElementById('etp_imap_host').value);
                formData.append('port', document.getElementById('etp_imap_port').value);
                formData.append('user', document.getElementById('etp_imap_user').value);
                formData.append('pass', document.getElementById('etp_imap_pass').value);
                formData.append('encryption', document.getElementById('etp_imap_encryption').value);

                try {
                    const response = await fetch('<?php echo admin_url('admin-ajax.php'); ?>', { method: 'POST', body: formData });
                    const result = await response.json();
                    
                    resultDiv.textContent = result.data.message;
                    resultDiv.classList.add(result.success ? 'success' : 'error');
                    if (!result.success && result.data.help) {
                        helpDiv.innerHTML = result.data.help;
                        helpDiv.style.display = 'block';
                    }
                } catch (error) {
                    resultDiv.textContent = 'Ein unerwarteter Fehler ist aufgetreten.';
                    resultDiv.classList.add('error');
                } finally {
                    spinner.style.visibility = 'hidden';
                    resultDiv.style.display = 'block';
                }
            });
        }
    });
    </script>
    <?php
}

